/*
  server_postgres_full_and_modular.js
  - This textdoc contains TWO things:
    1) FULL_SERVER: a single-file, near-complete conversion of your original server.js to PostgreSQL
       (uses `pg` Pool, schema `exelon_lms`, parameterized queries, async/await, and the same endpoints).
    2) MODULAR_GUIDE: a recommended modular refactor (files and code snippets) that splits the server
       into controllers, services, routes, and an auditLogger adapted for Postgres.

  Notes:
  - The uploaded SQLite file path available in this environment: /mnt/data/database.db
    Use the previously generated SQL at: /mnt/data/database_postgres_with_schema.sql to import into Postgres.
  - Replace environment variables in a .env file: DATABASE_URL, JWT_SECRET, NODE_ENV, PORT
  - This file is intentionally self-contained so you can paste and run after installing dependencies.

  To run (quick test):
    1) npm install express pg csv-parser multer bcryptjs jsonwebtoken cors dotenv
    2) Import the SQL (psql "$DATABASE_URL" -f /mnt/data/database_postgres_with_schema.sql)
    3) Create a .env with DATABASE_URL and JWT_SECRET
    4) node server_postgres_full_and_modular.js

  ---------- PART 1: FULL_SERVER (single-file) ----------
*/

// server.mjs (top portion)
import express, { json } from "express";
import cors from "cors";
import csv from "csv-parser";
import { existsSync, mkdirSync, createReadStream, unlinkSync } from "fs";
import path, { join } from "path";
import { fileURLToPath } from "url";
import pgPkg from "pg";
import bcrypt from "bcryptjs";
import jwt from "jsonwebtoken";
import multer from "multer";
const app = express();
// ESM-safe: derive __filename and __dirname
const __filename = fileURLToPath(import.meta.url);
const __dirname = path.dirname(__filename);
// ***** SERVE CLIENT BUILD *****
const publicPath = path.join(__dirname, 'public');
app.use(express.static(publicPath));

// SPA fallback → always return index.html for unknown routes
app.get('*', (req, res) => {
  res.sendFile(path.join(publicPath, 'index.html'));
});


// pg Pool (pg default import then destructure)
const { Pool } = pgPkg;

// bcrypt and jwt: default import then destructure
const { compare, hash: _hash } = bcrypt;
const { sign, verify } = jwt;

// Audit logger (must be an ESM module exporting `logAudit`)
import { logAudit } from "./auditLogger.mjs";


const port = process.env.PORT || 3001;
const JWT_SECRET = process.env.JWT_SECRET || "dev_jwt_secret";
const SCHEMA = "exelon_lms";
import 'dotenv/config';



const pool = new Pool({
  host: process.env.PGHOST, 
  port: process.env.PGPORT ,
  user: process.env.PGUSER ,
  password: process.env.PGPASSWORD ,
  database: process.env.PGDATABASE ,
    ssl: { rejectUnauthorized: false }
});
 console.log("process.env.PGHOST ========= : ",process.env.PGHOST)
// multer upload directory (use __dirname derived above)
const uploadDir = join(__dirname, "uploads");
if (!existsSync(uploadDir)) mkdirSync(uploadDir, { recursive: true });
const upload = multer({ dest: uploadDir });

// parse JSON
app.use(json());
//app.use(cors());

// ...rest of your server code

app.use(
  cors({
    origin: "*",
    methods: "GET,HEAD,PUT,PATCH,POST,DELETE",
    allowedHeaders: "Content-Type, Authorization",
  })
);
//app.use(json());

// serve static for Vite production (client/dist)
if (process.env.NODE_ENV === "production") {
  app.use(join(__dirname, "..", "client", "dist"));
}

async function query(text, params) {
  const client = await pool.connect();
  try {
    return await client.query(text, params);
  } finally {
    client.release();
  }
}

// Auth middleware
const authenticateToken = (req, res, next) => {
  const authHeader = req.headers["authorization"];
  const token = authHeader && authHeader.split(" ")[1];
  if (!token) return res.sendStatus(401);
  verify(token, JWT_SECRET, (err, user) => {
    if (err) return res.sendStatus(403);
    req.user = user;
    next();
  });
};

const isAdmin = (req, res, next) => {
  if (!req.user) return res.status(401).json({ error: "Not authenticated" });
  const r = (req.user.role || "").toLowerCase();
  if (r !== "administrator" && r !== "admin")
    return res.status(403).json({ error: "Forbidden" });
  next();
};

/* ----------------- FULL SERVER ENDPOINTS (converted) ----------------- */

// GET /api/employees
app.get("/api/employees", authenticateToken, async (req, res) => {
  try {
    const sql = `
      SELECT e.*,
        (SELECT special_leave_type FROM ${SCHEMA}.leave_logs WHERE employee_gaid = e.gaid AND leave_type = 'Special Leave' ORDER BY applied_date DESC LIMIT 1) as last_special_leave_type
      FROM ${SCHEMA}.employees e
    `;
    const { rows } = await query(sql, []);
    res.json(rows);
  } catch (err) {
    console.error(err);
    res.status(500).json({ error: err.message });
  }
});

// POST /api/employees
app.post("/api/employees", authenticateToken, async (req, res) => {
  const { gaid, name, doj, Team } = req.body;
  const username = req.user.username;
  if (!gaid || !name || !doj)
    return res.status(400).json({ error: "GAID, Name, and DOJ are required" });

  const client = await pool.connect();
  try {
    await client.query("BEGIN");
    await client.query(
      `INSERT INTO ${SCHEMA}.employees (gaid, name, date_of_joining, "Team") VALUES ($1,$2,$3,$4)`,
      [gaid, name, doj, Team]
    );
    await logAudit(
      username,
      "INSERT",
      "employees",
      gaid,
      null,
      null,
      JSON.stringify({ gaid, name, doj, Team })
    );

    // pro-rated
    let clCredit = 0,
      elCredit = 0;
    const joiningDate = new Date(doj);
    const joiningDay = joiningDate.getDate();
    if (joiningDay >= 1 && joiningDay <= 10) clCredit = 1;
    else if (joiningDay >= 11 && joiningDay <= 20) clCredit = 0.5;
    if (joiningDay >= 1 && joiningDay <= 6) elCredit = 1.5;
    else if (joiningDay >= 7 && joiningDay <= 15) elCredit = 1;
    else if (joiningDay >= 16 && joiningDay <= 20) elCredit = 0.5;

    if (clCredit > 0 || elCredit > 0) {
      const creditDate = new Date().toISOString();
      const currentMonth =
        new Date().getFullYear() +
        "-" +
        String(new Date().getMonth() + 1).padStart(2, "0");
      await client.query(
        `UPDATE ${SCHEMA}.employees SET total_cl_credited = COALESCE(total_cl_credited,0) + $1, total_el_credited = COALESCE(total_el_credited,0) + $2 WHERE gaid = $3`,
        [clCredit, elCredit, gaid]
      );
      await client.query(
        `INSERT INTO ${SCHEMA}.leave_credit_log (employee_gaid, credit_month, cl_credited, el_credited, credit_date) VALUES ($1,$2,$3,$4,$5)`,
        [gaid, currentMonth, clCredit, elCredit, creditDate]
      );
      await logAudit(
        username,
        "INSERT",
        "leave_credit_log",
        null,
        null,
        null,
        JSON.stringify({ gaid, currentMonth, clCredit, elCredit })
      );
    }

    await client.query("COMMIT");
    res
      .status(201)
      .json({
        message: "Employee added and initial leaves credited successfully",
      });
  } catch (err) {
    await client.query("ROLLBACK");
    console.error("Error adding employee:", err);
    res.status(500).json({ error: err.message });
  } finally {
    client.release();
  }
});

// Bulk upload
app.post(
  "/api/employees/bulk-upload",
  authenticateToken,
  upload.single("file"),
  async (req, res) => {
    const username = req.user.username;
    if (!req.file) return res.status(400).json({ error: "No file uploaded." });
    const filePath = req.file.path;
    const results = [];
    try {
      await new Promise((resolve, reject) => {
        createReadStream(filePath)
          .pipe(csv())
          .on("data", (data) => results.push(data))
          .on("end", resolve)
          .on("error", reject);
      });

      const client = await pool.connect();
      const addedGaids = [],
        updatedGaids = [],
        skipped = [];
      try {
        await client.query("BEGIN");
        for (const row of results) {
          const GAID = row.GAID || row.gaid;
          const Name = row.Name || row.name;
          const Date_of_joining = row.Date_of_joining || row.date_of_joining;
          const Team = row.Team || row.team || null;
          if (!GAID || !Name || !Date_of_joining) {
            skipped.push(row);
            continue;
          }
          const { rows: existing } = await client.query(
            `SELECT gaid FROM ${SCHEMA}.employees WHERE gaid = $1`,
            [GAID]
          );
          if (existing.length) {
            await client.query(
              `UPDATE ${SCHEMA}.employees SET Team = $1 WHERE gaid = $2`,
              [Team, GAID]
            );
            updatedGaids.push(GAID);
          } else {
            await client.query(
              `INSERT INTO ${SCHEMA}.employees (gaid, name, date_of_joining, Team) VALUES ($1,$2,$3,$4)`,
              [GAID, Name, Date_of_joining, Team]
            );
            addedGaids.push(GAID);
          }
        }
        await client.query("COMMIT");
        await logAudit(
          username,
          "BULK_UPLOAD",
          "employees",
          null,
          null,
          null,
          JSON.stringify({ added: addedGaids, updated: updatedGaids, skipped }),
          `Bulk uploaded ${addedGaids.length} employees, updated ${updatedGaids.length}.`
        );
        res.json({
          message: `Bulk upload successful. Added: ${addedGaids.length}. Updated: ${updatedGaids.length}. Skipped: ${skipped.length}.`,
          added: addedGaids,
          updated: updatedGaids,
          skipped,
        });
      } catch (err) {
        await client.query("ROLLBACK");
        throw err;
      } finally {
        client.release();
      }
    } catch (err) {
      console.error("Bulk upload error:", err);
      res.status(500).json({ error: err.message });
    } finally {
      try {
        unlinkSync(filePath);
      } catch (e) {}
    }
  }
);

// EL Carry Forward from CSV (expects el_carry_forward.csv in project root)
app.post(
  "/api/employees/el-carry-forward-process",
  authenticateToken,
  async (req, res) => {
    const username = req.user.username;
    const filePath = join(__dirname, "..", "el_carry_forward.csv");
    if (!existsSync(filePath))
      return res
        .status(404)
        .json({
          error: "el_carry_forward.csv not found in the root directory.",
        });
    const results = [];
    try {
      await new Promise((resolve, reject) => {
        createReadStream(filePath)
          .pipe(csv())
          .on("data", (d) => results.push(d))
          .on("end", resolve)
          .on("error", reject);
      });
      const updatedGaids = [],
        notFoundGaids = [];
      const client = await pool.connect();
      try {
        await client.query("BEGIN");
        for (const row of results) {
          const GAID = row.GAID || row.gaid;
          const el_carry_forward = row.el_carry_forward;
          if (!GAID || el_carry_forward === undefined) continue;
          const { rows: existing } = await client.query(
            `SELECT gaid FROM ${SCHEMA}.employees WHERE gaid = $1`,
            [GAID]
          );
          if (existing.length) {
            await client.query(
              `UPDATE ${SCHEMA}.employees SET el_carry_forward = $1 WHERE gaid = $2`,
              [el_carry_forward, GAID]
            );
            updatedGaids.push(GAID);
          } else {
            notFoundGaids.push(GAID);
          }
        }
        await client.query("COMMIT");
        await logAudit(
          username,
          "EL_CARRY_FORWARD_UPLOAD",
          "employees",
          null,
          null,
          null,
          JSON.stringify({ updated: updatedGaids, notFound: notFoundGaids }),
          `Processed EL Carry Forward file. Updated: ${updatedGaids.length}. Not Found: ${notFoundGaids.length}.`
        );
        res.json({
          message: `EL Carry Forward process successful. Updated: ${updatedGaids.length}. Not Found: ${notFoundGaids.length}.`,
          updated: updatedGaids,
          notFound: notFoundGaids,
        });
      } catch (err) {
        await client.query("ROLLBACK");
        throw err;
      } finally {
        client.release();
      }
    } catch (err) {
      console.error(err);
      res.status(500).json({ error: err.message });
    }
  }
);

// EL Credited Upload from CSV
app.post(
  "/api/employees/el-credited-upload",
  authenticateToken,
  async (req, res) => {
    const username = req.user.username;
    const filePath = join(__dirname, "..", "el_credited.csv");
    if (!existsSync(filePath))
      return res
        .status(404)
        .json({ error: "el_credited.csv not found in the root directory." });
    const results = [];
    try {
      await new Promise((resolve, reject) => {
        createReadStream(filePath)
          .pipe(csv())
          .on("data", (d) => results.push(d))
          .on("end", resolve)
          .on("error", reject);
      });
      const updatedGaids = [],
        notFoundGaids = [];
      const client = await pool.connect();
      try {
        await client.query("BEGIN");
        for (const row of results) {
          const GAID = row.GAID || row.gaid;
          const el_credited = row.el_credited;
          if (!GAID || el_credited === undefined) continue;
          const { rows: existing } = await client.query(
            `SELECT gaid FROM ${SCHEMA}.employees WHERE gaid = $1`,
            [GAID]
          );
          if (existing.length) {
            await client.query(
              `UPDATE ${SCHEMA}.employees SET total_el_credited = $1 WHERE gaid = $2`,
              [el_credited, GAID]
            );
            updatedGaids.push(GAID);
          } else {
            notFoundGaids.push(GAID);
          }
        }
        await client.query("COMMIT");
        await logAudit(
          username,
          "EL_CREDITED_UPLOAD",
          "employees",
          null,
          null,
          null,
          JSON.stringify({ updated: updatedGaids, notFound: notFoundGaids }),
          `Processed EL Credited file. Updated: ${updatedGaids.length}. Not Found: ${notFoundGaids.length}.`
        );
        res.json({
          message: `EL Credited process successful. Updated: ${updatedGaids.length}. Not Found: ${notFoundGaids.length}.`,
          updated: updatedGaids,
          notFound: notFoundGaids,
        });
      } catch (err) {
        await client.query("ROLLBACK");
        throw err;
      } finally {
        client.release();
      }
    } catch (err) {
      console.error(err);
      res.status(500).json({ error: err.message });
    }
  }
);

// DELETE employee
app.delete("/api/employees/:gaid", authenticateToken, async (req, res) => {
  const { gaid } = req.params;
  const username = req.user.username;
  if (!gaid) return res.status(400).json({ error: "GAID is required" });
  const client = await pool.connect();
  try {
    await client.query("BEGIN");
    const { rows } = await client.query(
      `SELECT * FROM ${SCHEMA}.employees WHERE gaid = $1`,
      [gaid]
    );
    if (!rows.length) {
      await client.query("ROLLBACK");
      return res.status(404).json({ error: "Employee not found" });
    }
    const employeeToDelete = rows[0];
    await client.query(`DELETE FROM ${SCHEMA}.employees WHERE gaid = $1`, [
      gaid,
    ]);
    await logAudit(
      username,
      "DELETE",
      "employees",
      gaid,
      null,
      JSON.stringify(employeeToDelete)
    );
    await client.query("COMMIT");
    res.json({ message: `Successfully Removed Employee: ${gaid}` });
  } catch (err) {
    await client.query("ROLLBACK");
    console.error(err);
    res.status(500).json({ error: err.message });
  } finally {
    client.release();
  }
});

// UPDATE employee (relieving_date, total_ual_taken)
app.put(
  "/api/employees/:gaid",
  authenticateToken,
  isAdmin,
  async (req, res) => {
    const { gaid } = req.params;
    const { relieving_date, total_ual_taken } = req.body;
    const username = req.user.username;
    const client = await pool.connect();
    try {
      await client.query("BEGIN");
      const { rows: oldRows } = await client.query(
        `SELECT relieving_date, total_ual_taken FROM ${SCHEMA}.employees WHERE gaid = $1`,
        [gaid]
      );
      const oldEmployee = oldRows[0] || {
        relieving_date: null,
        total_ual_taken: 0,
      };
      let updateFields = [],
        updateParams = [];
      if (relieving_date !== undefined) {
        updateFields.push("relieving_date");
        updateParams.push(relieving_date === "" ? null : relieving_date);
        if (
          oldEmployee.relieving_date !==
          (relieving_date === "" ? null : relieving_date)
        )
          await logAudit(
            username,
            "UPDATE",
            "employees",
            gaid,
            "relieving_date",
            oldEmployee.relieving_date,
            relieving_date === "" ? null : relieving_date
          );
      }
      if (total_ual_taken !== undefined) {
        updateFields.push("total_ual_taken");
        updateParams.push(total_ual_taken);
        if (oldEmployee.total_ual_taken !== total_ual_taken)
          await logAudit(
            username,
            "UPDATE",
            "employees",
            gaid,
            "total_ual_taken",
            oldEmployee.total_ual_taken,
            total_ual_taken
          );
      }
      if (updateFields.length > 0) {
        const setClause = updateFields
          .map((f, i) => `${f} = $${i + 1}`)
          .join(", ");
        await client.query(
          `UPDATE ${SCHEMA}.employees SET ${setClause} WHERE gaid = $${
            updateParams.length + 1
          }`,
          [...updateParams, gaid]
        );
      }

      // immediate adjustments for relieving_date in current month
      if (relieving_date) {
        const relievingDateObj = new Date(relieving_date);
        const relievingMonthStr =
          relievingDateObj.getFullYear() +
          "-" +
          String(relievingDateObj.getMonth() + 1).padStart(2, "0");
        const currentMonthStr =
          new Date().getFullYear() +
          "-" +
          String(new Date().getMonth() + 1).padStart(2, "0");
        if (relievingMonthStr === currentMonthStr) {
          const { rows: existingCreditRows } = await client.query(
            `SELECT * FROM ${SCHEMA}.leave_credit_log WHERE employee_gaid = $1 AND credit_month = $2`,
            [gaid, relievingMonthStr]
          );
          if (existingCreditRows.length) {
            const existingCreditLog = existingCreditRows[0];
            let entitledCl = 0,
              entitledEl = 0;
            const relievingDay = relievingDateObj.getDate();
            if (relievingDay >= 11 && relievingDay <= 20) entitledCl = 0.5;
            else if (relievingDay >= 21) entitledCl = 1;
            if (relievingDay >= 7 && relievingDay <= 15) entitledEl = 0.5;
            else if (relievingDay >= 16 && relievingDay <= 20) entitledEl = 1;
            else if (relievingDay >= 21) entitledEl = 1.5;
            const clAdjustment = entitledCl - existingCreditLog.cl_credited;
            const elAdjustment = entitledEl - existingCreditLog.el_credited;
            if (clAdjustment !== 0 || elAdjustment !== 0) {
              await client.query(
                `UPDATE ${SCHEMA}.employees SET total_cl_credited = total_cl_credited + $1, total_el_credited = total_el_credited + $2 WHERE gaid = $3`,
                [clAdjustment, elAdjustment, gaid]
              );
              await client.query(
                `UPDATE ${SCHEMA}.leave_credit_log SET cl_credited = $1, el_credited = $2 WHERE id = $3`,
                [entitledCl, entitledEl, existingCreditLog.id]
              );
            }
          }
        }
      }

      await client.query("COMMIT");
      res.json({ message: "Relieving date updated successfully." });
    } catch (err) {
      await client.query("ROLLBACK");
      console.error(err);
      res.status(500).json({ error: err.message });
    } finally {
      client.release();
    }
  }
);

// Helper calculateLeaveDays (same algorithm)
function calculateLeaveDaysSync(fromDate, toDate, dayType, holidays) {
  let leaveDays = 0;
  let currentDate = new Date(fromDate);
  const endDate = new Date(toDate);
  while (currentDate <= endDate) {
    const dayOfWeek = currentDate.getDay();
    const isWeekend = dayOfWeek === 0 || dayOfWeek === 6;
    const formattedDate = currentDate.toISOString().split("T")[0];
    const isHoliday = holidays.some((h) => h.date === formattedDate);
    if (!isWeekend && !isHoliday) leaveDays++;
    currentDate.setDate(currentDate.getDate() + 1);
  }
  if (dayType === "Half Day") return leaveDays * 0.5;
  return leaveDays;
}

// Add leave log (complete)
app.post(
  "/api/employees/:gaid/leave-log",
  authenticateToken,
  async (req, res) => {
    const { gaid } = req.params;
    const { fromDate, toDate, leaveType, dayType, remarks, specialLeaveType } =
      req.body;
    let { calculatedDays } = req.body;
    const username = req.user.username;
    if (!fromDate || !toDate || !leaveType || !dayType)
      return res
        .status(400)
        .json({ error: "Missing required leave log fields." });

    try {
      const { rows: holidaysRows } = await query(
        `SELECT date FROM ${SCHEMA}.holidays`,
        []
      );
      const holidays = holidaysRows.map((r) => ({ date: r.date }));
      calculatedDays = calculateLeaveDaysSync(
        fromDate,
        toDate,
        dayType,
        holidays
      );

      // overlapping check
      const { rows: overlapRows } = await query(
        `SELECT * FROM ${SCHEMA}.leave_logs WHERE employee_gaid = $1 AND from_date <= $2 AND to_date >= $3 LIMIT 1`,
        [gaid, toDate, fromDate]
      );
      if (overlapRows.length) {
        const overlappingLeave = overlapRows[0];
        let leaveDescription = overlappingLeave.leave_type;
        if (
          overlappingLeave.leave_type === "Special Leave" &&
          overlappingLeave.special_leave_type
        )
          leaveDescription = overlappingLeave.special_leave_type;
        return res
          .status(400)
          .json({
            error: `The selected dates overlap with an existing ${leaveDescription} from ${overlappingLeave.from_date} to ${overlappingLeave.to_date}.`,
          });
      }

      // Paternity/Maternity/Bereavement checks (fetch employee)
      const { rows: empRows } = await query(
        `SELECT date_of_joining, paternity_leave_count FROM ${SCHEMA}.employees WHERE gaid = $1`,
        [gaid]
      );
      const employee = empRows[0];
      if (!employee)
        return res.status(404).json({ error: "Employee not found." });

      if (
        leaveType === "Special Leave" &&
        specialLeaveType === "Paternity Leave"
      ) {
        const doj = new Date(employee.date_of_joining);
        const leaveFromDate = new Date(fromDate);
        const serviceDays = Math.floor(
          (leaveFromDate - doj) / (1000 * 60 * 60 * 24)
        );
        if (serviceDays < 90)
          return res
            .status(400)
            .json({
              error:
                "Paternity Leave can only be applied after 90 days of service.",
            });
        if (calculatedDays > 3)
          return res
            .status(400)
            .json({
              error:
                "Paternity Leave cannot be applied for more than 3 days at a time.",
            });
        if ((employee.paternity_leave_count || 0) >= 2)
          return res
            .status(400)
            .json({
              error: "Paternity Leave can only be applied twice in service.",
            });
      }

      if (
        leaveType === "Special Leave" &&
        specialLeaveType === "Maternity Leave"
      ) {
        const doj = new Date(employee.date_of_joining);
        const leaveFromDate = new Date(fromDate);
        const serviceDays = Math.floor(
          (leaveFromDate - doj) / (1000 * 60 * 60 * 24)
        );
        if (serviceDays < 90)
          return res
            .status(400)
            .json({
              error:
                "Maternity Leave can only be applied after 90 days of service.",
            });
        if (calculatedDays > 180)
          return res
            .status(400)
            .json({
              error:
                "Maternity Leave cannot be applied for more than 180 days at a time.",
            });
      }

      if (leaveType === "Bereavement Leave" && calculatedDays > 3)
        return res
          .status(400)
          .json({
            error:
              "Bereavement Leave can only be applied for exactly 3 days at a time.",
          });

      // Insert leave log and update counts in transaction
      const client = await pool.connect();
      try {
        await client.query("BEGIN");
        const insertRes = await client.query(
          `INSERT INTO ${SCHEMA}.leave_logs (employee_gaid, from_date, to_date, leave_type, day_type, days, remarks, applied_date, special_leave_type) VALUES ($1,$2,$3,$4,$5,$6,$7,$8,$9) RETURNING id`,
          [
            gaid,
            fromDate,
            toDate,
            leaveType,
            dayType,
            calculatedDays,
            remarks,
            new Date().toISOString(),
            specialLeaveType,
          ]
        );
       
        
        const leaveLogId = insertRes.rows[0].id;
         console.log("Calling here -------------------------------",insertRes)
        await logAudit(
          username,
          "INSERT",
          "leave_logs",
          leaveLogId,
          null,
          null,
          JSON.stringify(req.body)
        );
        // determine leave column
        let leaveColumn = null;
        if (leaveType === "Special Leave") {
          if (specialLeaveType === "Paternity Leave")
            leaveColumn = "total_paternity_leave_taken";
          else if (specialLeaveType === "Marriage Leave")
            leaveColumn = "total_marriage_leave_taken";
          else leaveColumn = "total_special_leave_taken";
        } else {
          switch (leaveType) {
            case "Casual Leave":
              leaveColumn = "total_cl_taken";
              break;
            case "Earned Leave":
              leaveColumn = "total_el_taken";
              break;
            case "Loss Of Pay":
              leaveColumn = "lop_taken";
              break;
            case "Bereavement Leave":
              leaveColumn = "total_bereavement_leave_taken";
              break;
            case "Unauthorized Leave":
              leaveColumn = "total_ual_taken";
              break;
          }
        }

        if (leaveColumn) {
          const { rows: oldRows2 } = await client.query(
            `SELECT ${leaveColumn} FROM ${SCHEMA}.employees WHERE gaid = $1`,
            [gaid]
          );
          const oldLeaveValue = oldRows2.length
            ? oldRows2[0][leaveColumn] || 0
            : 0;
          await client.query(
            `UPDATE ${SCHEMA}.employees SET ${leaveColumn} = COALESCE(${leaveColumn},0) + $1 WHERE gaid = $2`,
            [calculatedDays, gaid]
          );
          await logAudit(
            username,
            "UPDATE",
            "employees",
            gaid,
            leaveColumn,
            oldLeaveValue,
            oldLeaveValue + calculatedDays
          );
        }

        if (
          leaveType === "Special Leave" &&
          specialLeaveType === "Paternity Leave"
        ) {
          const { rows: pRows } = await client.query(
            `SELECT paternity_leave_count FROM ${SCHEMA}.employees WHERE gaid = $1`,
            [gaid]
          );
          const oldPaternityCount = pRows.length
            ? pRows[0].paternity_leave_count || 0
            : 0;
          await client.query(
            `UPDATE ${SCHEMA}.employees SET paternity_leave_count = COALESCE(paternity_leave_count,0) + 1 WHERE gaid = $1`,
            [gaid]
          );
          await logAudit(
            username,
            "UPDATE",
            "employees",
            gaid,
            "paternity_leave_count",
            oldPaternityCount,
            oldPaternityCount + 1
          );
        }

        // Retroactive maternity deduction
        if (
          leaveType === "Special Leave" &&
          specialLeaveType === "Maternity Leave"
        ) {
          try {
            const startDate = new Date(fromDate);
            const maternityStartMonth =
              startDate.getFullYear() +
              "-" +
              String(startDate.getMonth() + 1).padStart(2, "0");
            const endDate = new Date(toDate);
            let current = new Date(
              startDate.getFullYear(),
              startDate.getMonth(),
              1
            );
            while (current <= endDate) {
              const creditMonth =
                current.getFullYear() +
                "-" +
                String(current.getMonth() + 1).padStart(2, "0");
              if (creditMonth === maternityStartMonth) {
                current.setMonth(current.getMonth() + 1);
                continue;
              }
              const { rows: logRows } = await client.query(
                `SELECT * FROM ${SCHEMA}.leave_credit_log WHERE employee_gaid = $1 AND credit_month = $2`,
                [gaid, creditMonth]
              );
              if (logRows.length) {
                const log = logRows[0];
                const { rows: oldEmp } = await client.query(
                  `SELECT total_cl_credited, total_el_credited FROM ${SCHEMA}.employees WHERE gaid = $1`,
                  [gaid]
                );
                const oldCl = oldEmp.length
                  ? oldEmp[0].total_cl_credited || 0
                  : 0;
                const oldEl = oldEmp.length
                  ? oldEmp[0].total_el_credited || 0
                  : 0;
                await client.query(
                  `UPDATE ${SCHEMA}.employees SET total_cl_credited = total_cl_credited - $1, total_el_credited = total_el_credited - $2 WHERE gaid = $3`,
                  [log.cl_credited, log.el_credited, gaid]
                );
                await logAudit(
                  username,
                  "UPDATE",
                  "employees",
                  gaid,
                  "total_cl_credited",
                  oldCl,
                  oldCl - log.cl_credited,
                  `Maternity deduction for ${creditMonth}`
                );
                await logAudit(
                  username,
                  "UPDATE",
                  "employees",
                  gaid,
                  "total_el_credited",
                  oldEl,
                  oldEl - log.el_credited,
                  `Maternity deduction for ${creditMonth}`
                );
                await client.query(
                  `DELETE FROM ${SCHEMA}.leave_credit_log WHERE id = $1`,
                  [log.id]
                );
                await logAudit(
                  username,
                  "DELETE",
                  "leave_credit_log",
                  log.id,
                  null,
                  JSON.stringify(log),
                  null,
                  `Maternity deduction for ${creditMonth}`
                );
              }
              current.setMonth(current.getMonth() + 1);
            }
          } catch (e) {
            console.error("Error during retroactive maternity deduction:", e);
          }
        }

        await client.query("COMMIT");
        res
          .status(201)
          .json({
            id: insertRes && insertRes.rows ? insertRes.rows[0].id : null,
          });
      } catch (err) {
        await client.query("ROLLBACK");
        throw err;
      } finally {
        client.release();
      }
    } catch (err) {
      console.error("Error in leave-log:", err);
      res.status(500).json({ error: err.message });
    }
  }
);

// Leave history
app.get(
  "/api/employees/:gaid/leave-history",
  authenticateToken,
  async (req, res) => {
    const { gaid } = req.params;
    const { year, month } = req.query;
    if (!year || !month)
      return res.status(400).json({ error: "Year and month are required." });
    const paddedMonth = month.toString().padStart(2, "0");
    const startDate = `${year}-${paddedMonth}-01`;
    const endDate = new Date(year, month, 0).getDate();
    const endOfMonth = `${year}-${paddedMonth}-${endDate}`;
    try {
      const { rows } = await query(
        `SELECT * FROM ${SCHEMA}.leave_logs WHERE employee_gaid = $1 AND from_date BETWEEN $2 AND $3 ORDER BY from_date DESC`,
        [gaid, startDate, endOfMonth]
      );
      res.json(rows);
    } catch (err) {
      console.error(err);
      res.status(500).json({ error: err.message });
    }
  }
);

// Delete leave logs (bulk)
app.delete("/api/leave-logs", authenticateToken, async (req, res) => {
  const { ids } = req.body;
  const username = req.user.username;
  if (!ids || !Array.isArray(ids) || ids.length === 0)
    return res
      .status(400)
      .json({ error: "An array of leave log IDs is required." });
  const client = await pool.connect();
  try {
    await client.query("BEGIN");
    for (const id of ids) {
      const { rows: logRows } = await client.query(
        `SELECT * FROM ${SCHEMA}.leave_logs WHERE id = $1`,
        [id]
      );
      if (!logRows.length) continue;
      const log = logRows[0];
      await logAudit(
        username,
        "DELETE",
        "leave_logs",
        log.id,
        null,
        JSON.stringify(log)
      );
      let leaveColumn = null;
      let decrementPaternity = false;
      if (log.leave_type === "Special Leave") {
        if (log.special_leave_type === "Paternity Leave") {
          leaveColumn = "total_paternity_leave_taken";
          decrementPaternity = true;
        } else if (log.special_leave_type === "Marriage Leave")
          leaveColumn = "total_marriage_leave_taken";
        else leaveColumn = "total_special_leave_taken";
      } else {
        switch (log.leave_type) {
          case "Casual Leave":
            leaveColumn = "total_cl_taken";
            break;
          case "Earned Leave":
            leaveColumn = "total_el_taken";
            break;
          case "Loss Of Pay":
            leaveColumn = "lop_taken";
            break;
          case "Bereavement Leave":
            leaveColumn = "total_bereavement_leave_taken";
            break;
          default:
            leaveColumn = null;
        }
      }
      if (!leaveColumn) continue; // nothing to update
      const { rows: empRows } = await client.query(
        `SELECT ${leaveColumn}, paternity_leave_count FROM ${SCHEMA}.employees WHERE gaid = $1`,
        [log.employee_gaid]
      );
      const oldLeaveValue = empRows.length ? empRows[0][leaveColumn] || 0 : 0;
      const oldPaternityCount = empRows.length
        ? empRows[0].paternity_leave_count || 0
        : 0;
      await client.query(
        `UPDATE ${SCHEMA}.employees SET ${leaveColumn} = COALESCE(${leaveColumn},0) - $1${
          decrementPaternity
            ? ", paternity_leave_count = COALESCE(paternity_leave_count,0) - 1"
            : ""
        } WHERE gaid = $2`,
        [log.days, log.employee_gaid]
      );
      await logAudit(
        username,
        "UPDATE",
        "employees",
        log.employee_gaid,
        leaveColumn,
        oldLeaveValue,
        oldLeaveValue - log.days
      );
      if (decrementPaternity)
        await logAudit(
          username,
          "UPDATE",
          "employees",
          log.employee_gaid,
          "paternity_leave_count",
          oldPaternityCount,
          oldPaternityCount - 1
        );
    }
    const placeholders = ids.map((_, i) => `$${i + 1}`).join(",");
    await client.query(
      `DELETE FROM ${SCHEMA}.leave_logs WHERE id IN (${placeholders})`,
      ids
    );
    await client.query("COMMIT");
    res.json({ deleted: ids.length });
  } catch (err) {
    await client.query("ROLLBACK");
    console.error(err);
    res.status(500).json({ error: err.message });
  } finally {
    client.release();
  }
});

// Authentication / Users
app.post("/api/login", async (req, res) => {
  console.log("its logiN-----------------");
  const { username, password } = req.body;
  try {
    const { rows } = await query(
      `SELECT * FROM ${SCHEMA}.users WHERE username = $1`,
      [username]
    );
    console.log("user detail = ",rows)
    if (!rows.length)
      return res.status(400).json({ error: "Invalid username or password" });
    const user = rows[0];
    const match = await compare(password, user.password);
    if (!match)
      return res.status(400).json({ error: "Invalid username or password" });
    const token = sign(
      { id: user.id, username: user.username, role: user.role },
      JWT_SECRET,
      { expiresIn: "8h" }
    );
    res.json({
      token,
      user: {
        id: user.id,
        username: user.username,
        role: user.role,
        displayName: user.displayusername || user.displayusername,
      },
    });
  } catch (err) {
    console.error(err);
    res.status(500).json({ error: err.message });
  }
});

app.get("/api/users", authenticateToken, isAdmin, async (req, res) => {
  try {
    const { rows } = await query(
      `SELECT id, username, role, displayusername  FROM ${SCHEMA}.users`
    );
    res.json(rows);
  } catch (err) {
    console.error(err);
    res.status(500).json({ error: err.message });
  }
});

app.post("/api/users", authenticateToken, isAdmin, async (req, res) => {
  const { username, password, role, displayName } = req.body;
  const auditorUsername = req.user.username;
  try {
    const hash = await _hash(password, 10);
    const { rows } = await query(
      `INSERT INTO exelon_lms.users(
	username, password, role, displayusername)
	VALUES ($1,$2,$3,$4) RETURNING id`,
      [username, hash, role, displayName]
    );
    const newUserId = rows[0].id;
    await logAudit(
      auditorUsername,
      "INSERT",
      "users",
      newUserId,
      null,
      null,
      JSON.stringify({ username, role, displayName })
    );
    res.json({ id: newUserId });
  } catch (err) {
    console.error(err);
    res.status(400).json({ error: err.message });
  }
});

app.delete("/api/users/:id", authenticateToken, isAdmin, async (req, res) => {
  const userIdToDelete = req.params.id;
  const auditorUsername = req.user.username;
  const client = await pool.connect();
  try {
    await client.query("BEGIN");
    const { rows } = await client.query(
      `SELECT id, username, role ,displayusername FROM ${SCHEMA}.users WHERE id = $1`,
      [userIdToDelete]
    );
    if (!rows.length) {
      await client.query("ROLLBACK");
      return res.status(404).json({ error: "User not found" });
    }
    await client.query(`DELETE FROM ${SCHEMA}.users WHERE id = $1`, [
      userIdToDelete,
    ]);
    await logAudit(
      auditorUsername,
      "DELETE",
      "users",
      userIdToDelete,
      null,
      JSON.stringify(rows[0])
    );
    await client.query("COMMIT");
    res.json({ deleted: 1 });
  } catch (err) {
    await client.query("ROLLBACK");
    console.error(err);
    res.status(500).json({ error: err.message });
  } finally {
    client.release();
  }
});

app.put(
  "/api/users/:id/reset-password",
  authenticateToken,
  isAdmin,
  async (req, res) => {
    const userId = req.params.id;
    const { password } = req.body;
    const auditorUsername = req.user.username;
    try {
      const hash = await _hash(password, 10);
      const { rows } = await query(
        `SELECT password FROM ${SCHEMA}.users WHERE id = $1`,
        [userId]
      );
      const oldPasswordHash = rows.length ? "hashed" : null;
      await query(`UPDATE ${SCHEMA}.users SET password = $1 WHERE id = $2`, [
        hash,
        userId,
      ]);
      await logAudit(
        auditorUsername,
        "UPDATE",
        "users",
        userId,
        "password",
        oldPasswordHash,
        "hashed"
      );
      res.json({ updated: 1 });
    } catch (err) {
      console.error(err);
      res.status(500).json({ error: err.message });
    }
  }
);

app.put(
  "/api/users/:id/change-role",
  authenticateToken,
  isAdmin,
  async (req, res) => {
    const userId = req.params.id;
    const { role } = req.body;
    const auditorUsername = req.user.username;
    try {
      const { rows } = await query(
        `SELECT role FROM ${SCHEMA}.users WHERE id = $1`,
        [userId]
      );
      const oldRole = rows.length ? rows[0].role : null;
      await query(`UPDATE ${SCHEMA}.users SET role = $1 WHERE id = $2`, [
        role,
        userId,
      ]);
      if (oldRole !== role)
        await logAudit(
          auditorUsername,
          "UPDATE",
          "users",
          userId,
          "role",
          oldRole,
          role
        );
      res.json({ updated: 1 });
    } catch (err) {
      console.error(err);
      res.status(500).json({ error: err.message });
    }
  }
);

// Holidays endpoints
app.get("/api/holidays", authenticateToken, async (req, res) => {
  try {
    const { rows } = await query(
      `SELECT * FROM ${SCHEMA}.holidays ORDER BY date ASC`,
      []
    );
    res.json(rows);
  } catch (err) {
    console.error(err);
    res.status(500).json({ error: err.message });
  }
});

app.post(
  "/api/holidays/upload",
  authenticateToken,
  isAdmin,
  upload.single("file"),
  async (req, res) => {
    const username = req.user.username;
    if (!req.file) return res.status(400).json({ error: "No file uploaded." });
    const filePath = req.file.path;
    const holidays = [];
    try {
      await new Promise((resolve, reject) => {
        createReadStream(filePath)
          .pipe(csv())
          .on("data", (d) => holidays.push(d))
          .on("end", resolve)
          .on("error", reject);
      });
      unlinkSync(filePath);
      const validHolidays = holidays.filter(
        (h) => h["Date"] && h["Holiday name"]
      );
      if (validHolidays.length === 0)
        return res
          .status(400)
          .json({
            error:
              'CSV has no valid holiday data. Required columns are "Date" and "Holiday name".',
          });
      const formatDateForDB = (dateString) => {
        const parts = dateString.split("-");
        const day = parts[0];
        const month =
          new Date(Date.parse(parts[1] + " 1, 2000")).getMonth() + 1;
        const year = `20${parts[2]}`;
        return `${year}-${String(month).padStart(2, "0")}-${day}`;
      };
      const client = await pool.connect();
      try {
        await client.query("BEGIN");
        const { rows: oldHolidays } = await client.query(
          `SELECT * FROM ${SCHEMA}.holidays`
        );
        await client.query(`DELETE FROM ${SCHEMA}.holidays`);
        await logAudit(
          username,
          "BULK_DELETE",
          "holidays",
          null,
          null,
          JSON.stringify(oldHolidays),
          null,
          `Cleared ${oldHolidays.length} existing holidays before upload.`
        );
        const insertPromises = validHolidays.map((h) =>
          client.query(
            `INSERT INTO ${SCHEMA}.holidays (date, name) VALUES ($1,$2)`,
            [formatDateForDB(h["Date"]), h["Holiday name"]]
          )
        );
        await Promise.all(insertPromises);
        await client.query("COMMIT");
        await logAudit(
          username,
          "BULK_INSERT",
          "holidays",
          null,
          null,
          null,
          JSON.stringify(validHolidays),
          `Uploaded ${validHolidays.length} new holidays.`
        );
        res.json({
          message: `Successfully uploaded ${validHolidays.length} holidays.`,
        });
      } catch (err) {
        await client.query("ROLLBACK");
        throw err;
      } finally {
        client.release();
      }
    } catch (err) {
      try {
        unlinkSync(filePath);
      } catch (e) {}
      console.error(err);
      res.status(500).json({ error: err.message });
    }
  }
);

// creditLeaves function (monthly credit job)
async function creditLeaves(auditorUsername = "SYSTEM_SCHEDULER") {
  console.log("Running monthly leave credit job...");
  const today = new Date();
  const currentMonth =
    today.getFullYear() + "-" + String(today.getMonth() + 1).padStart(2, "0");
  const startOfCurrentMonth =
    today.getFullYear() +
    "-" +
    String(today.getMonth() + 1).padStart(2, "0") +
    "-01";
  try {
    const { rows: employees } = await query(
      `SELECT * FROM ${SCHEMA}.employees WHERE relieving_date IS NULL OR relieving_date >= $1`,
      [startOfCurrentMonth]
    );
    for (const employee of employees) {
      const { rows: hasBeenCreditedRows } = await query(
        `SELECT * FROM ${SCHEMA}.leave_credit_log WHERE employee_gaid = $1 AND credit_month = $2`,
        [employee.gaid, currentMonth]
      );
      if (hasBeenCreditedRows.length) {
        console.log(
          `Employee ${employee.gaid} has already been credited for ${currentMonth}.`
        );
        continue;
      }
      let clCredit = 0,
        elCredit = 0,
        leaveLogicApplied = false;
      const endOfCurrentMonth = new Date(
        today.getFullYear(),
        today.getMonth() + 1,
        0
      )
        .toISOString()
        .split("T")[0];
      const { rows: maternityRows } = await query(
        `SELECT from_date FROM ${SCHEMA}.leave_logs WHERE employee_gaid = $1 AND leave_type = 'Special Leave' AND special_leave_type = 'Maternity Leave' AND from_date <= $2 AND to_date >= $3 LIMIT 1`,
        [employee.gaid, endOfCurrentMonth, startOfCurrentMonth]
      );
      if (maternityRows.length) {
        const maternityStartDate = new Date(maternityRows[0].from_date);
        const maternityStartMonth =
          maternityStartDate.getFullYear() +
          "-" +
          String(maternityStartDate.getMonth() + 1).padStart(2, "0");
        if (maternityStartMonth === currentMonth) {
          clCredit = 1;
          elCredit = 1.5;
        } else {
          clCredit = 0;
          elCredit = 0;
        }
        leaveLogicApplied = true;
      }
      if (!leaveLogicApplied) {
        const joiningDate = new Date(employee.date_of_joining);
        const joiningMonth =
          joiningDate.getFullYear() +
          "-" +
          String(joiningDate.getMonth() + 1).padStart(2, "0");
        if (joiningMonth === currentMonth) {
          const joiningDay = joiningDate.getDate();
          if (joiningDay >= 1 && joiningDay <= 10) clCredit = 1;
          else if (joiningDay >= 11 && joiningDay <= 20) clCredit = 0.5;
          if (joiningDay >= 1 && joiningDay <= 6) elCredit = 1.5;
          else if (joiningDay >= 7 && joiningDay <= 15) elCredit = 1;
          else if (joiningDay >= 16 && joiningDay <= 20) elCredit = 0.5;
        } else if (employee.relieving_date) {
          const relievingDate = new Date(employee.relieving_date);
          const relievingMonth =
            relievingDate.getFullYear() +
            "-" +
            String(relievingDate.getMonth() + 1).padStart(2, "0");
          if (relievingMonth === currentMonth) {
            const relievingDay = relievingDate.getDate();
            if (relievingDay >= 11 && relievingDay <= 20) clCredit = 0.5;
            else if (relievingDay >= 21) clCredit = 1;
            if (relievingDay >= 7 && relievingDay <= 15) elCredit = 0.5;
            else if (relievingDay >= 16 && relievingDay <= 20) elCredit = 1;
            else if (relievingDay >= 21) elCredit = 1.5;
          } else {
            clCredit = 1;
            elCredit = 1.5;
          }
        } else {
          clCredit = 1;
          elCredit = 1.5;
        }
      }
      if (clCredit > 0 || elCredit > 0) {
        const client = await pool.connect();
        try {
          await client.query("BEGIN");
          const { rows: oldEmp } = await client.query(
            `SELECT total_cl_credited, total_el_credited FROM ${SCHEMA}.employees WHERE gaid = $1`,
            [employee.gaid]
          );
          const oldCl = oldEmp.length ? oldEmp[0].total_cl_credited || 0 : 0;
          const oldEl = oldEmp.length ? oldEmp[0].total_el_credited || 0 : 0;
          await client.query(
            `UPDATE ${SCHEMA}.employees SET total_cl_credited = COALESCE(total_cl_credited,0) + $1, total_el_credited = COALESCE(total_el_credited,0) + $2 WHERE gaid = $3`,
            [clCredit, elCredit, employee.gaid]
          );
          await client.query(
            `INSERT INTO ${SCHEMA}.leave_credit_log (employee_gaid, credit_month, cl_credited, el_credited, credit_date) VALUES ($1,$2,$3,$4,$5)`,
            [
              employee.gaid,
              currentMonth,
              clCredit,
              elCredit,
              new Date().toISOString(),
            ]
          );
          await logAudit(
            auditorUsername,
            "UPDATE",
            "employees",
            employee.gaid,
            "total_cl_credited",
            oldCl,
            oldCl + clCredit,
            `Monthly credit for ${currentMonth}`
          );
          await logAudit(
            auditorUsername,
            "UPDATE",
            "employees",
            employee.gaid,
            "total_el_credited",
            oldEl,
            oldEl + elCredit,
            `Monthly credit for ${currentMonth}`
          );
          await logAudit(
            auditorUsername,
            "INSERT",
            "leave_credit_log",
            null,
            null,
            null,
            JSON.stringify({
              employee_gaid: employee.gaid,
              credit_month: currentMonth,
              cl_credited: clCredit,
              el_credited: elCredit,
            }),
            `Monthly credit for ${currentMonth}`
          );
          await client.query("COMMIT");
          console.log(
            `Credited ${clCredit} CL and ${elCredit} EL to employee ${employee.gaid} for ${currentMonth}.`
          );
        } catch (err) {
          await client.query("ROLLBACK");
          console.error(err);
        } finally {
          client.release();
        }
      }
    }
    console.log("Monthly leave credit job finished.");
  } catch (err) {
    console.error("Error during monthly leave credit job:", err);
  }
}

// Scheduler (setInterval checking every minute) - keep as-is but using new creditLeaves
(function scheduleJobs() {
  let lastMonthlyCreditRun = null;
  let lastAnnualCarryForwardRun = null;
  setInterval(async () => {
    const now = new Date();
    const currentMinute = now.getMinutes();
    const currentHour = now.getHours();
    const currentDay = now.getDate();
    const currentMonth = now.getMonth() + 1;
    const currentYear = now.getFullYear();
    const monthlyRunId = currentYear + "-" + currentMonth;
    if (currentDay === 1 && currentHour === 2 && currentMinute === 0) {
      if (lastMonthlyCreditRun !== monthlyRunId) {
        console.log("Scheduler: running monthly credit");
        creditLeaves("SYSTEM_SCHEDULER");
        lastMonthlyCreditRun = monthlyRunId;
      }
    }
    const annualRunId = currentYear;
    if (
      currentMonth === 1 &&
      currentDay === 1 &&
      currentHour === 0 &&
      currentMinute === 2
    ) {
      if (lastAnnualCarryForwardRun !== annualRunId) {
        console.log("Scheduler: running annual carry forward");
        // perform annual carry-forward
        try {
          const employeesBefore = (
            await query(
              `SELECT gaid, total_el_credited, total_el_taken, el_carry_forward FROM ${SCHEMA}.employees`
            )
          ).rows;
          await query(
            `UPDATE ${SCHEMA}.employees SET total_el_credited = CASE WHEN (COALESCE(total_el_credited,0) - COALESCE(total_el_taken,0)) > 40 THEN 40 ELSE (COALESCE(total_el_credited,0) - COALESCE(total_el_taken,0)) END, total_el_taken = 0, el_carry_forward = CASE WHEN (COALESCE(total_el_credited,0) - COALESCE(total_el_taken,0)) > 40 THEN 40 ELSE (COALESCE(total_el_credited,0) - COALESCE(total_el_taken,0)) END`
          );
          const employeesAfter = (
            await query(
              `SELECT gaid, total_el_credited, total_el_taken, el_carry_forward FROM ${SCHEMA}.employees`
            )
          ).rows;
          await logAudit(
            "SYSTEM_SCHEDULER",
            "BULK_UPDATE",
            "employees",
            null,
            null,
            JSON.stringify(employeesBefore),
            JSON.stringify(employeesAfter),
            `Annual EL carry forward process.`
          );
        } catch (e) {
          console.error("Error in annual carry forward:", e);
        }
        lastAnnualCarryForwardRun = annualRunId;
      }
    }
  }, 60000);
  console.log("Started custom job scheduler (checks every minute).");
})();

// Manual trigger for credit
app.post("/api/leaves/credit", authenticateToken, isAdmin, async (req, res) => {
  try {
    await creditLeaves(req.user.username);
    res.json({ message: "Leave credit process completed successfully." });
  } catch (err) {
    console.error(err);
    res.status(500).json({ error: err.message });
  }
});

// Undo credit
app.post(
  "/api/leaves/undo-credit",
  authenticateToken,
  isAdmin,
  async (req, res) => {
    console.log("Received request to undo leave credit.");
    const username = req.user.username;
    const client = await pool.connect();
    try {
      await client.query("BEGIN");
      const { rows: lastCreditRows } = await client.query(
        `SELECT credit_month FROM ${SCHEMA}.leave_credit_log ORDER BY id DESC LIMIT 1`
      );
      if (!lastCreditRows.length) {
        await client.query("ROLLBACK");
        return res
          .status(404)
          .json({ message: "No leave credit transaction found to undo." });
      }
      const { credit_month } = lastCreditRows[0];
      const logs = (
        await client.query(
          `SELECT * FROM ${SCHEMA}.leave_credit_log WHERE credit_month = $1`,
          [credit_month]
        )
      ).rows;
      if (!logs.length) {
        await client.query("ROLLBACK");
        return res
          .status(404)
          .json({ message: `No credit logs found for month ${credit_month}.` });
      }
      for (const log of logs) {
        const { rows: empRows } = await client.query(
          `SELECT total_cl_credited, total_el_credited FROM ${SCHEMA}.employees WHERE gaid = $1`,
          [log.employee_gaid]
        );
        const oldCl = empRows.length ? empRows[0].total_cl_credited || 0 : 0;
        const oldEl = empRows.length ? empRows[0].total_el_credited || 0 : 0;
        await client.query(
          `UPDATE ${SCHEMA}.employees SET total_cl_credited = COALESCE(total_cl_credited,0) - $1, total_el_credited = COALESCE(total_el_credited,0) - $2 WHERE gaid = $3`,
          [log.cl_credited, log.el_credited, log.employee_gaid]
        );
        await logAudit(
          username,
          "UPDATE",
          "employees",
          log.employee_gaid,
          "total_cl_credited",
          oldCl,
          oldCl - log.cl_credited,
          `Undo credit for ${credit_month}`
        );
        await logAudit(
          username,
          "UPDATE",
          "employees",
          log.employee_gaid,
          "total_el_credited",
          oldEl,
          oldEl - log.el_credited,
          `Undo credit for ${credit_month}`
        );
      }
      await client.query(
        `DELETE FROM ${SCHEMA}.leave_credit_log WHERE credit_month = $1`,
        [credit_month]
      );
      await logAudit(
        username,
        "BULK_DELETE",
        "leave_credit_log",
        null,
        null,
        JSON.stringify(logs),
        null,
        `Undo credit for ${credit_month}`
      );
      await client.query("COMMIT");
      res.json({
        message: `Successfully undid leave credit for ${credit_month}.`,
      });
    } catch (err) {
      await client.query("ROLLBACK");
      console.error(err);
      res.status(500).json({ error: err.message });
    } finally {
      client.release();
    }
  }
);

// Get credited for month
app.get(
  "/api/leaves/credited-for-month",
  authenticateToken,
  async (req, res) => {
    const { month, gaid } = req.query;
    if (!month || !gaid)
      return res
        .status(400)
        .json({ error: "Month and gaid parameters are required" });
    try {
      const { rows } = await query(
        `SELECT SUM(cl_credited) as total_cl_credited, SUM(el_credited) as total_el_credited FROM ${SCHEMA}.leave_credit_log WHERE credit_month = $1 AND employee_gaid = $2`,
        [month, gaid]
      );
      res.json(rows[0]);
    } catch (err) {
      console.error(err);
      res.status(500).json({ error: err.message });
    }
  }
);

// Export single employee leave log as CSV
app.get(
  "/api/employees/:gaid/export-leave-log",
  authenticateToken,
  async (req, res) => {
    const { gaid } = req.params;
    try {
      const { rows: empRows } = await query(
        `SELECT name FROM ${SCHEMA}.employees WHERE gaid = $1`,
        [gaid]
      );
      if (!empRows.length)
        return res.status(404).json({ error: "Employee not found." });
      const employee = empRows[0];
      const { rows: leaveLogs } = await query(
        `SELECT from_date, to_date, leave_type, day_type, days, remarks, applied_date, special_leave_type FROM ${SCHEMA}.leave_logs WHERE employee_gaid = $1 ORDER BY applied_date DESC`,
        [gaid]
      );
      let csvData = [];
      let headers = [
        "Leave Type",
        "From Date",
        "To Date",
        "Day Type",
        "Days",
        "Remarks",
        "Applied Date",
      ];
      const hasSpecialLeave = leaveLogs.some(
        (l) => l.leave_type === "Special Leave"
      );
      if (hasSpecialLeave) headers.splice(1, 0, "Special Leave Type");
      csvData.push(headers.join(","));
      leaveLogs.forEach((log) => {
        let row = [
          `"${log.leave_type}"`,
          `"${log.from_date}"`,
          `"${log.to_date}"`,
          `"${log.day_type}"`,
          log.days,
          `"${log.remarks || ""}"`,
          `"${log.applied_date}"`,
        ];
        if (hasSpecialLeave)
          row.splice(1, 0, `"${log.special_leave_type || ""}"`);
        csvData.push(row.join(","));
      });
      res.setHeader("Content-Type", "text/csv");
      res.setHeader(
        "Content-Disposition",
        `attachment; filename="${employee.name.replace(
          / /g,
          "_"
        )}_Leave_Log.csv"`
      );
      res.send(csvData.join("\n"));
    } catch (err) {
      console.error(err);
      res.status(500).json({ error: err.message });
    }
  }
);

// Export all leave balance
app.get(
  "/api/export-all-leave-balance",
  authenticateToken,
  async (req, res) => {
    try {
      const { rows: employees } = await query(
        `SELECT gaid, name, total_cl_credited, total_cl_taken, total_el_credited, total_el_taken, el_carry_forward FROM ${SCHEMA}.employees`,
        []
      );
      let csvData = [];
      const headers = [
        "GAID",
        "Name",
        "CL Credited",
        "CL Taken",
        "CL Balance",
        "EL Credited",
        "EL Taken",
        "EL Balance",
        "EL Carry Forward",
      ];
      csvData.push(headers.join(","));
      employees.forEach((emp) => {
        const cl_balance =
          (emp.total_cl_credited || 0) - (emp.total_cl_taken || 0);
        const el_balance =
          (emp.total_el_credited || 0) - (emp.total_el_taken || 0);
        const row = [
          `"${emp.gaid}"`,
          `"${emp.name}"`,
          emp.total_cl_credited || 0,
          emp.total_cl_taken || 0,
          cl_balance,
          emp.total_el_credited || 0,
          emp.total_el_taken || 0,
          el_balance,
          emp.el_carry_forward || 0,
        ];
        csvData.push(row.join(","));
      });
      res.setHeader("Content-Type", "text/csv");
      res.setHeader(
        "Content-Disposition",
        'attachment; filename="all_employee_leave_balance.csv"'
      );
      res.send(csvData.join("\n"));
    } catch (err) {
      console.error(err);
      res.status(500).json({ error: err.message });
    }
  }
);

// Export all leave history
app.get(
  "/api/export-all-leave-history",
  authenticateToken,
  async (req, res) => {
    try {
      const { rows: leaveLogs } = await query(
        `SELECT e.gaid, e.name, ll.leave_type, ll.special_leave_type, ll.from_date, ll.to_date, ll.days, ll.remarks, ll.applied_date FROM ${SCHEMA}.leave_logs ll JOIN ${SCHEMA}.employees e ON ll.employee_gaid = e.gaid ORDER BY e.gaid, ll.from_date`,
        []
      );
      let csvData = [];
      const headers = [
        "GAID",
        "Name",
        "Leave Type",
        "Special Leave Type",
        "From Date",
        "To Date",
        "Days",
        "Remarks",
        "Applied Date",
      ];
      csvData.push(headers.join(","));
      leaveLogs.forEach((l) => {
        const row = [
          `"${l.gaid}"`,
          `"${l.name}"`,
          `"${l.leave_type}"`,
          `"${l.special_leave_type || ""}"`,
          `"${l.from_date}"`,
          `"${l.to_date}"`,
          l.days,
          `"${l.remarks || ""}"`,
          `"${l.applied_date}"`,
        ];
        csvData.push(row.join(","));
      });
      res.setHeader("Content-Type", "text/csv");
      res.setHeader(
        "Content-Disposition",
        'attachment; filename="all_employee_leave_history.csv"'
      );
      res.send(csvData.join("\n"));
    } catch (err) {
      console.error(err);
      res.status(500).json({ error: err.message });
    }
  }
);

// Delete SYSTEM_SCHEDULER audit logs
app.delete(
  "/api/audit-logs/system-scheduler",
  authenticateToken,
  isAdmin,
  async (req, res) => {
    const username = req.user.username;
    try {
      const { rows: systemLogsToDelete } = await query(
        `SELECT * FROM ${SCHEMA}.audit_logs WHERE username = $1`,
        ["SYSTEM_SCHEDULER"]
      );
      if (!systemLogsToDelete.length)
        return res.json({
          message: "No SYSTEM_SCHEDULER logs found to delete.",
        });
      await query(`DELETE FROM ${SCHEMA}.audit_logs WHERE username = $1`, [
        "SYSTEM_SCHEDULER",
      ]);
      await logAudit(
        username,
        "BULK_DELETE",
        "audit_logs",
        null,
        null,
        JSON.stringify(systemLogsToDelete),
        null,
        `Deleted ${systemLogsToDelete.length} SYSTEM_SCHEDULER logs.`
      );
      res.json({
        message: `Successfully deleted ${systemLogsToDelete.length} SYSTEM_SCHEDULER logs.`,
      });
    } catch (err) {
      console.error(err);
      res.status(500).json({ error: err.message });
    }
  }
);

// GET audit logs with filters
app.get("/api/audit-logs", authenticateToken, async (req, res) => {
  const { table_name, record_id, username } = req.query;
  let queryStr = `SELECT * FROM ${SCHEMA}.audit_logs WHERE 1=1`;
  const params = [];
  let idx = 1;
  if (table_name) {
    queryStr += ` AND table_name = $${idx++}`;
    params.push(table_name);
  }
  if (record_id) {
    queryStr += ` AND record_id = $${idx++}`;
    params.push(record_id);
  }
  if (username) {
    queryStr += ` AND username = $${idx++}`;
    params.push(username);
  }
  queryStr += ` ORDER BY timestamp DESC`;
  try {
    const { rows } = await query(queryStr, params);
    res.json(rows);
  } catch (err) {
    console.error(err);
    res.status(500).json({ error: err.message });
  }
});

// //clear audit
app.delete("/api/deleteallauditlogs", authenticateToken, async (req, res) => {
  try{
     const { table_name, record_id, username } = req.query;
  await query(`DELETE FROM ${SCHEMA}.audit_logs`);
  await logAudit(
    username,
    "BULK_DELETE",
    "audit_logs",
    null,
    null,
    "Deleted all logs.",
    null,
    `Deleted all logs.`
  );
  res.json({
    statusCode:200,
    message: `Successfully deleted All logs.`,
  });
  }catch(err){
    res.json({
    statusCode:400,
    message: `Successfully deleted All logs.`,
  });
  }
});

// Serve SPA in production
app.get("*", (req, res) => {
  if (process.env.NODE_ENV === "production")
    res.sendFile(join(__dirname, "..", "client", "dist", "index.html"));
  else res.status(404).send("Not found");
});

app.listen(port, () =>
  console.log(`Server listening at http://localhost:${port}`)
);



/* ---------- PART 2: MODULAR_GUIDE (recommended refactor) ----------
   Instead of a single huge file, split into:

   /server
     app.js                <-- express app & middleware
     /routes
       employees.js
       users.js
       leaves.js
       holidays.js
       audit.js
     /controllers
       employeesController.js
       usersController.js
       leavesController.js
     /services
       employeeService.js  <-- DB calls using pg pool
       leaveService.js
       userService.js
     auditLogger.js        <-- implements logAudit using pg
     db.js                 <-- exports configured Pool

   Example: db.js
   ----------------
   const { Pool } = require('pg');
   const pool = new Pool({ connectionString: process.env.DATABASE_URL });
   module.exports = pool;

   Example: auditLogger.js
   ------------------------
   const pool = require('./db');
   async function logAudit(username, action, table, record_id, column, oldValue, newValue, extra) {
     try {
       await pool.query(`INSERT INTO ${SCHEMA}.audit_logs (username, action, table_name, record_id, column_name, old_value, new_value, extra, timestamp) VALUES ($1,$2,$3,$4,$5,$6,$7,$8,NOW())`, [username, action, table, record_id, column, oldValue, newValue, extra]);
     } catch (e) { console.error('Failed to log audit:', e); }
   }
   module.exports = { logAudit };

   Example: employeesController.js
   --------------------------------
   const express = require('express');
   const router = express.Router();
   const employeeService = require('../services/employeeService');
   router.get('/', async (req,res)=>{ const employees = await employeeService.getAll(); res.json(employees); });
   // ... other routes ...
   module.exports = router;

   This modular approach makes the codebase far easier to test and maintain. If you want, I can generate
   the full modular file set (all route files, controllers, services, db.js, auditLogger.js) as separate
   text documents or in a zip. Tell me which you prefer.
*/

// End of file
